package mybeaverbot;

import cz.cuni.pogamut.Client.Agent;
import java.util.*;
import cz.cuni.pogamut.MessageObjects.*;
import java.util.logging.Logger;



/**
 * Private map using Floyd-Warshall for pathfinding.
 * @author Martin Krulis
 */
public class WarshallMap
{
	protected int badEdge;	// Flag mask representing unusable edge.
	
	protected Logger log = null;
	protected Agent agent = null;
	
	// Hash table converting navPoint IDs to our own indices.
	protected Map<Integer, Integer> navPointIndices = null;
	
	// Warshall's matrix of distances.
	protected float[][] distances;
	
	
	/**
	 * Checks whether the edge is usable.
	 * @param from Starting nav point.
	 * @param edge NeighNav object representing the edge.
	 * @return boolean
	 */
	protected boolean checkEdge(NavPoint from, NeighNav edge)
	{
		// Bad flags (prohibited edges, swimming, flying...).
		if ((edge.flags & badEdge) != 0) return false;
		
		// Lift flags.
		if ((edge.flags & NeighNavFlag.SPECIAL.get()) != 0) return true;
		
		// Check whether the climbing angle is not so steep.
		if ((Triple.distanceInPlane(from.location, edge.neighbour.location) < (edge.neighbour.location.z - from.location.z)) &&
			(Triple.distanceInSpace(from.location, edge.neighbour.location) > 100))
			return false;
			
		// Check whether the jump is not so high.
		if (((edge.flags & NeighNavFlag.JUMP.get()) != 0) && (edge.neighbour.location.z - from.location.z > 80))
			return false;
		
		return true;
	}
	
	
	/**
	 * Constructor performs Floyd-Warshall's algorithm.
	 * @param log Reference to user's log.
	 * @param agent Reference to bot's (agent's) object.
	 * @param navPoints All nav points in the map.
	 */
	public WarshallMap(Logger log, Agent agent, ArrayList<NavPoint> navPoints)
	{
		this.log = log;
		this.agent = agent;

		// Prepares bad edge flag for checkEdge method.
		badEdge = NeighNavFlag.FLY.get() | NeighNavFlag.LADDER.get() | 	NeighNavFlag.PROSCRIBED.get() |
			NeighNavFlag.SWIM.get() | NeighNavFlag.PLAYERONLY.get() | NeighNavFlag.FORCED.get();
		
		// Initialize navPoint indices mapping.
		navPointIndices = new HashMap<Integer, Integer>();
		for (int i = 0; i < navPoints.size(); i++)
			navPointIndices.put(navPoints.get(i).getID(), i);

		// Initialize distance matrix.
		int size = navPoints.size();
		distances = new float[size][size];
		for (int i = 0; i < size; i++)
			for (int j = 0; j < size; j++)
				distances[i][j] = (i == j) ? 0.0f : Float.POSITIVE_INFINITY;

		// Set edge lengths into distance matrix.
		for (int i = 0; i < size; i++) {
			Triple location = navPoints.get(i).location;
			for (NeighNav neigh : navPoints.get(i).neighbours)
				if (checkEdge(navPoints.get(i), neigh))
					distances[i][navPointIndices.get(neigh.neighbour.getID())] = (float)Triple.distanceInSpace(location, neigh.neighbour.getLocation());
		}
		
		// Perform Floyd-Warshall.
		for (int k = 0; k < size; k++)
			for (int i = 0; i < size; i++)
				for (int j = 0; j < size; j++) {
					float newLen = distances[i][k] + distances[k][j];
					if (distances[i][j] > newLen) distances[i][j] = newLen;
				}
		
		// Log unreachable paths.
		int count = 0;
		for (int i = 0; i < size; i++)
			for (int j = 0; j < size; j++)
				if (distances[i][j] == Float.POSITIVE_INFINITY) {
					log.finer("Unreachable path from " + navPoints.get(i).UnrealID + "to" + navPoints.get(j).UnrealID);
					count++;
				}
		log.info("There are " + count + " unreachable nav point pairs.");
	}
	

	/**
	 * Checks whether a nav point is reachable from bot's position.
	 * @return boolean
	 */
	public boolean isReachable(NavPoint navPoint)
	{
		if ((navPoint == null) || !navPoint.reachable) return false;
		Triple location = agent.getMemory().getAgentLocation();
		if ((Triple.distanceInPlane(location, navPoint.location) < (navPoint.location.z - location.z)) &&
			(Triple.distanceInSpace(location, navPoint.location) > 80))
			return false;
		return true;
	}
	
	
	/**
	 * Calculate's distance between two nav points (using pathfinding).
	 * @return Distance or POSITIVE_INFINITY if there's no path.
	 */
	public float getDistance(NavPoint from, NavPoint to)
	{
		if ((from == null) || (to == null)) return Float.POSITIVE_INFINITY;
		int idxFrom = navPointIndices.get(from.getID());
		int idxTo = navPointIndices.get(to.getID());
		return distances[idxFrom][idxTo];
	}
	
	
	/**
	 * Calculates path using precomputed Warshall matrix.
	 * @param res Array where the results will be placed.
	 * @return True if the path was found, false otherwise.
	 */
	public boolean getPath(NavPoint from, NavPoint to, ArrayList<NavPoint> res)
	{
		if (res == null) return false;
	
		log.finer("Calculating path from " + from.UnrealID + " to " + to.UnrealID);
			
		res.clear();
		res.add(from);
		int toIdx = navPointIndices.get(to.getID());
		while (from.getID() != to.getID()) {
			int fromIdx = navPointIndices.get(from.getID());

			NavPoint next = null;
			float minDist = Float.POSITIVE_INFINITY;
			for (NeighNav edge : from.neighbours) {
				int idx = navPointIndices.get(edge.neighbour.getID());
				float dist = distances[fromIdx][idx] + distances[idx][toIdx];
				if (checkEdge(from, edge) && (dist < minDist)) {
					minDist = dist;
					next = edge.neighbour;
				}
			}
			
			if ((next == null) || (minDist == Float.POSITIVE_INFINITY))
				return false;
			
			from = next;
			res.add(from);
		}
		
		return true;
	}
	
}